<?php
header("Content-Type: application/json");
header("Access-Control-Allow-Origin: *");

require_once 'db.php';
require_once 'castHelper.php'; 
try {
    if ($_SERVER['REQUEST_METHOD'] !== 'GET') {
        http_response_code(405);
        echo json_encode(["status" => "error", "message" => "Only GET method allowed"]);
        exit;
    }

    // Validate inputs
    $user_id = $_GET['user_id'] ?? null;
    $type = $_GET['type'] ?? null;
    $page = isset($_GET['page']) ? max((int)$_GET['page'], 1) : 1;
    $limit = isset($_GET['limit']) ? max((int)$_GET['limit'], 1) : 10;
    $offset = ($page - 1) * $limit;

    if (!$user_id || !in_array($type, ['followers', 'following'])) {
        http_response_code(400);
        echo json_encode(["status" => "error", "message" => "Missing or invalid parameters"]);
        exit;
    }

    // SQL & bindings
    if ($type === 'followers') {
        $count_sql = "SELECT COUNT(*) FROM user_follows WHERE following_id = :user_id";
        $data_sql = "
            SELECT u.user_id, u.name, u.username, u.profile_url
            FROM user_follows f
            JOIN users u ON u.user_id = f.follower_id
            WHERE f.following_id = :user_id
            ORDER BY f.id DESC
            LIMIT :limit OFFSET :offset
        ";
    } else { // following
        $count_sql = "SELECT COUNT(*) FROM user_follows WHERE follower_id = :user_id";
        $data_sql = "
            SELECT u.user_id, u.name, u.username, u.profile_url
            FROM user_follows f
            JOIN users u ON u.user_id = f.following_id
            WHERE f.follower_id = :user_id
            ORDER BY f.id DESC
            LIMIT :limit OFFSET :offset
        ";
    }

    // Get total count
    $count_stmt = $pdo->prepare($count_sql);
    $count_stmt->execute(['user_id' => $user_id]);
    $total = (int)$count_stmt->fetchColumn();

    // Get paginated results
    $data_stmt = $pdo->prepare($data_sql);
    $data_stmt->bindValue(':user_id', $user_id, PDO::PARAM_STR);
    $data_stmt->bindValue(':limit', $limit, PDO::PARAM_INT);
    $data_stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
    $data_stmt->execute();
    $users = $data_stmt->fetchAll(PDO::FETCH_ASSOC);
$users = array_map('castTypes', $users);
    echo json_encode([
        "status" => "success",
        "type" => $type,
        "user_id" => $user_id,
        "total" => $total,
        "page" => $page,
        "limit" => $limit,
        "pages" => ceil($total / $limit),
        "data" => $users
    ]);

} catch (Exception $e) {
    http_response_code(500);
    echo json_encode([
        "status" => "error",
        "message" => "Server error",
        "error" => $e->getMessage()
    ]);
}
?>
