<?php
header("Content-Type: application/json");
header("Access-Control-Allow-Origin: *");

require_once 'db.php';

// Pagination parameters
$page  = isset($_GET['page']) && is_numeric($_GET['page']) ? (int) $_GET['page'] : 1;
$limit = isset($_GET['limit']) && is_numeric($_GET['limit']) ? (int) $_GET['limit'] : 10;
$offset = ($page - 1) * $limit;

try {
    // Step 1: Get total count of course categories
    $countQuery = "SELECT COUNT(*) AS total FROM course_category WHERE status = 1";
    $countStmt = $pdo->prepare($countQuery);
    $countStmt->execute();
    $total = $countStmt->fetch(PDO::FETCH_ASSOC)['total'];

    // Step 2: Fetch paginated course categories
    $categoryQuery = "SELECT * FROM course_category WHERE status = 1 ORDER BY updated_at DESC LIMIT :limit OFFSET :offset";
    $categoryStmt = $pdo->prepare($categoryQuery);
    $categoryStmt->bindParam(':limit', $limit, PDO::PARAM_INT);
    $categoryStmt->bindParam(':offset', $offset, PDO::PARAM_INT);
    $categoryStmt->execute();
    $categories = $categoryStmt->fetchAll(PDO::FETCH_ASSOC);

    $finalResult = [];

    // Step 3: For each category, fetch related notes
    foreach ($categories as $category) {
        $c_category_id = $category['c_category_id'];

        $noteQuery = "
            SELECT 
                n.*, 
                COUNT(DISTINCT fc.comment_id) AS total_comments, 
                COUNT(DISTINCT l.like_id) AS total_likes
            FROM notes n
            LEFT JOIN feed_comments fc ON fc.feed_id = n.note_id
            LEFT JOIN likes l ON l.post_id = n.note_id
            WHERE n.home_show_cate = :c_category_id
            GROUP BY n.note_id
        ";

        $noteStmt = $pdo->prepare($noteQuery);
        $noteStmt->bindParam(':c_category_id', $c_category_id);
        $noteStmt->execute();
        $notes = $noteStmt->fetchAll(PDO::FETCH_ASSOC);

        $category['notes'] = $notes;
        $finalResult[] = $category;
    }

    // Step 4: Return structured response
    echo json_encode([
        "status" => "success",
        "page" => $page,
        "limit" => $limit,
        "total" => (int)$total,
        "pages" => ceil($total / $limit),
        "data" => $finalResult
    ]);
} catch (PDOException $e) {
    http_response_code(500);
    echo json_encode([
        "status" => "error",
        "message" => "Database error: " . $e->getMessage()
    ]);
}
?>
